#!/usr/bin/python3
"""
usage: ptuxversion [options] [<commit-ish>]

options:
  --check-development         return 0 if not released, otherwise 1
  --tag-match <pattern>       tag-match pattern ('*')
  --default-branch <branch>   default branch name (main)
"""

# Displays a repository's "version number", which is intended to be suitable
# for use as/synonymous with a package version number.
#
# Theory of Operation
# ===================
#
# We display a unique value intended to be used as the associated package's APT version.
# The value displayed is auto-generated from tag data, commit counts, and timestamps,
# the details of which provide some flexibility in defining the version number namespace.
# The goal is a version numbering scheme that looks visibly different for released vs.
# non-released packages, that works correctly during APT upgrades, and is usable within
# and across project workgroups without risk of collisions.
#
# The version-number-generation logic is as follows:
#
#   If there is a tag matching <pattern> at <commit-ish>, the tag (with vendor
#   portion removed, if any; see below) is printed: this is the package version.
#
#   If there isn't a tag at <commit-ish>, we display the nearest reachable tag in
#   the commit history, followed by a "+T" plus a UTC timestamp.
#
#   If there are no tags reachable from <commit-ish>, we display the count of
#   commits reachable from <commit-ish> by traveling backwards along <branch>.
#
#   Finally, we append "~g" and the <commit-ish>'s rev-parse hash.
#
# We rely heavily on git's rev-list --count output for all of the above.
#
#
# On Tag Patterns
# ---------------
#
# The recommended tagging pattern is "<vendor>/<count>", where:
#
#   <vendor> is a developer/customer-defined text string
#     /      is the literal '/' character
#   <count>  is the value reported by git rev-list --count along
#            the project's release branch
#
# Suitable tags under this recommendation include:
#
#   ptux/60
#   catee/60
#
# Other tagging patterns are possible and permitted, but in practice the
# alternatives have been shown to increase risk of ambiguity in package
# versions, additional developer effort during release transitions, and
# traceability errors.
#
#
# On Default Parameters/ptux.conf
# -------------------------------
#
# Older releases of ptuxversion assumed that the project's release branch was
# named "master", or "main", and we still assume the latter unless told otherwise.
#
# Different default values for --default-branch and --tag-match can be specified
# in a configuration file that ptuxversion will load at startup if present. The file
# must be named 'ptux.conf', and must be placed in the repository's debian/ directory.
#
# An example configuration file is in /usr/share/devscripts-ptux/ptux.conf, and is
# included with the devscripts-ptux source code.


import configparser
from docopt import docopt
from pathlib import Path
from ptuxutil import sh
from ptuxutil.sh.contrib import git
import string
import sys
import time


def _distance(commit, base):
    if base:
        arg = '{}..{}'.format(base, commit)
    else:
        arg = commit

    count = git('rev-list', '--count', arg)
    return int(count)


def _are_synonomous(c1, c2):
    diff = git('rev-list', '{}...{}'.format(c1, c2))
    return not bool(diff)


def _is_dirty():
    modified = git('ls-files', '--modified', '--others', '--exclude-standard')
    return bool(modified)


def _is_contained(commit, by):
    if not by: return False
    excess = git('rev-list', commit, '^{}'.format(by))
    return not bool(excess)


def _strip_vendor(tag):
    vendor, sep, basetag = tag.rpartition('/')
    return basetag


def _unmangle_tag(tag):
    'Unmangle a tag name according to DEP14'
    tab = {ord('%'): ord(':'), ord('_'): ord('~'), ord('#'): None}
    return tag.translate(tab)


def _nearest_tag(commit, limit, patterns):
    try:
        args = ['--tags']
        args += ['--abbrev=0']
        args += ['--match={}'.format(p) for p in patterns]
        args += [commit]
        tag = git.describe(*args).strip()
    except sh.ErrorReturnCode:
        return None

    if limit:
        if not _is_contained(limit, by=tag):
            return None

    return tag


def tag_plus_distance(commit, base=None, match=['*']):
    tag = _nearest_tag(commit, base, match)
    if tag:
        t = _strip_vendor(_unmangle_tag(tag))
        d = _distance(commit, tag)
        return '{}+{}'.format(t, d) if d else t
    else:
        return str(_distance(commit, base))


def _is_valid(commit):
    try:
        sha = git('rev-parse', '--verify', '{}^{{commit}}'.format(commit))
    except sh.ErrorReturnCode_128:
        return False

    return True

def _nearest_published(commit, branch, match):
    if _is_valid('origin/'+branch):
        base = git('merge-base', commit, 'origin/'+branch).strip()
    elif _is_valid(branch):
        base = git('merge-base', commit, branch).strip()
    else:
        base = None

    return _nearest_tag(commit, base, match) or base


def is_development(commit, branch, match):
    return (_are_synonomous(commit, 'HEAD') and _is_dirty()) \
           or not _is_contained(commit, _nearest_published(commit, branch, match))


def describe(commit='HEAD', branch='main', match=['*']):
    y = ''
    if _are_synonomous(commit, 'HEAD') and _is_dirty():
        y = '~dirty'

    n = _nearest_published(commit, branch, match)

    t = ''
    if y or not _is_contained(commit, n):
        sec_since_epoch = int(time.time())
        t = '+T{}'.format(sec_since_epoch)

    d = tag_plus_distance(commit=n, match=match) if n else '0'
    r = git('rev-parse', '--short', commit).strip()

    return '{}{}~g{}{}'.format(d, t, r, y)


def cli(argv=sys.argv[1:]):
    conf = configparser.ConfigParser()
    conf.read(str(git('rev-parse', '--show-toplevel')).strip() + '/debian/ptux.conf')

    defs = dict()

    try:
        for sect in ['DEFAULT', 'ptuxversion']:
            for key in conf[sect]:
                defs.update({'--' + key: conf[sect][key]})
    except KeyError:
        pass

    args = docopt(__doc__, argv=argv)
    args.update({k: v for k, v in defs.items() if args[k] is None})

    commit = args['<commit-ish>'] or 'HEAD'
    branch = args['--default-branch'] or 'main'
    match = [args['--tag-match'] or '*']

    if args['--check-development']:
        if is_development(commit, branch, match):
            sys.exit(0)
        else:
            sys.exit(1)

    print (describe(commit, branch, match))


if __name__ == "__main__":
    cli()
