/*
 * Prints an application-specific, machine-unique identifier.
 *
 * As described in machine-id(5) and systemd-id128(1), we must hash our private
 * machine-id with a constant, application-specific identifier when we wish to
 * produce a public, machine-unique identifer. The public, machine-unique
 * identifier, called an "application-id", is unique to us but does not reveal
 * the contents of our machine-id.
 *
 * We don't want to use our machine-id(5) to uniquely identify ourselves in an
 * untrusted network, because persons in possession of that value can, at least
 * in theory, remotely reproduce some calculated results that lead to security
 * risks. They might be able to reproduce the output of our PRNG, for example,
 * which would break the "S" in HTTPS.
 *
 * Our application-id is attached to all data sent offsite, so that viewers of
 * that data can tell what machine it came from.
 *
 * ref: machine-id(5), systemd-id128(1), and sd-id128(3).
 */

#include <stdio.h>
#include <systemd/sd-id128.h>

/*  $ systemd-id128 -p new
 *
 * Note: This value isn't a "secret" in the cryptographic sense.
 */
#define FRONTIER_APP_ID_0 SD_ID128_MAKE(5e,c7,38,9d,90,6f,4a,db,bb,14,56,10,3b,10,ea,e4)

int main(int argc, const char *argv[])
{
       sd_id128_t id;
       sd_id128_get_machine_app_specific(FRONTIER_APP_ID_0, &id);
       printf(SD_ID128_FORMAT_STR "\n", SD_ID128_FORMAT_VAL(id));
       return 0;
}



