/******************************************************************************/
/*  ARGS_MAIN.C                                                               */
/*                                                                            */
/* Copyright (c) 2003 Texas Instruments Incorporated                          */
/* http://www.ti.com/                                                         */
/*                                                                            */
/*  Redistribution and  use in source  and binary forms, with  or without     */
/*  modification,  are permitted provided  that the  following conditions     */
/*  are met:                                                                  */
/*                                                                            */
/*     Redistributions  of source  code must  retain the  above copyright     */
/*     notice, this list of conditions and the following disclaimer.          */
/*                                                                            */
/*     Redistributions in binary form  must reproduce the above copyright     */
/*     notice, this  list of conditions  and the following  disclaimer in     */
/*     the  documentation  and/or   other  materials  provided  with  the     */
/*     distribution.                                                          */
/*                                                                            */
/*     Neither the  name of Texas Instruments Incorporated  nor the names     */
/*     of its  contributors may  be used to  endorse or  promote products     */
/*     derived  from   this  software  without   specific  prior  written     */
/*     permission.                                                            */
/*                                                                            */
/*  THIS SOFTWARE  IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS     */
/*  "AS IS"  AND ANY  EXPRESS OR IMPLIED  WARRANTIES, INCLUDING,  BUT NOT     */
/*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR     */
/*  A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT     */
/*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,     */
/*  SPECIAL,  EXEMPLARY,  OR CONSEQUENTIAL  DAMAGES  (INCLUDING, BUT  NOT     */
/*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,     */
/*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY     */
/*  THEORY OF  LIABILITY, WHETHER IN CONTRACT, STRICT  LIABILITY, OR TORT     */
/*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE     */
/*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.      */
/*                                                                            */
/******************************************************************************/
/******************************************************************************/
/* The ARGS data structure is defined according to a convention with linker.  */
/*                                                                            */
/* If the user want to pass arguments to loader, "--args=###" option has to   */
/* be used in linking to generate executable. With this option, the linker    */
/* will allocate a section starting with __c_args__, and with this "###" many */
/* bytes. The loader when parses the arguments, will dump the number of       */
/* arguments, argc as the 1st arguments at address __c_args__, all the actual */
/* arguments will be dumped after that. The total space for these arguments   */
/* will not exceed "###" bytes.                                               */
/*                                                                            */
/* if "--args="###" is not used as a linker option, linker will put -1 at     */
/* __c_args__ location.                                                       */
/* Use _symval to load the 32-bit value corresponding to symbol __c_args__    */
/*                                                                            */
/* Based on the above convention, the following code implements the access to */
/* these arguments when main is called.                                       */
/*                                                                            */
/* This function is called from boot.asm or boot.c.                           */
/******************************************************************************/

#include <stdlib.h>

#ifdef _TMS320C6X
extern __far int main(int argc, char *argv[]);
#else
extern int main(int argc, char *argv[]);
#endif

typedef struct { int argc; char *argv[1]; } ARGS;

/******************************************************************************/
/* "Old" targets use the old mechanism of the linker defining __c_args__ to   */
/* -1 if there is no .args section. That method works for TI compilers, but   */
/* not for LLVM. LLVM assumes __c_args__ always has a valid address and       */
/* speculatively loads its value regardless if it is -1 or not. The solution  */
/* is to use a weak reference which will be 0 if no definition of __c_args__  */
/* exists. This works because the semantics of weak references are well       */
/* understood and cannot be hoisted. I chose not to update all toolchains in  */
/* case there are users relying on the current method. The TI proprietary     */
/* compilers will not hoist the load so there is no bug for those compilers.  */
/******************************************************************************/
#if (defined(__TMS320C2000__) && !defined(__TI_EABI__)) || \
   defined(__TMS320C6X__)                               || \
   defined(__MSP430__)                                  || \
   defined(__PRU__)                                     || \
   defined(__ARP32__)                                   || \
   defined(__TI_ARM__)
#define USE_WEAK_REF 0
#else
#define USE_WEAK_REF 1
#endif

#if USE_WEAK_REF
extern __attribute__((weak)) ARGS __c_args__;
#else
extern ARGS __c_args__;
#define NO_C_ARGS   ((unsigned long)-1)
#endif

#ifdef __TI_RTS_BUILD
/*---------------------------------------------------------------------------*/
/* __TI_default_args_main indicates that the default TI _args_main routine   */
/* is being used.  The linker makes assumptions about what _args_main does   */
/* when this symbol is seen.  This symbols should NOT be defined if a        */
/* customized _args_main routine is used.                                    */
/*---------------------------------------------------------------------------*/
__asm("__TI_default_args_main .set 1");
#endif

int _args_main()
{
   register int    argc = 0;
   register char **argv = 0;

#if USE_WEAK_REF
   if (&__c_args__ != NULL)
   {
      argc = __c_args__.argc;
      argv = __c_args__.argv;
   }
#else
#pragma diag_suppress 1107,173
    register ARGS *pargs = (ARGS*)_symval(&__c_args__);
#pragma diag_default 1107,173


   if (_symval(&__c_args__) != NO_C_ARGS) 
   {
      argc = pargs->argc;
      argv = pargs->argv;
   }
#endif

   return main(argc, argv);
}
