// TODO:

#include <fstream>
#include <iostream>
#include <mutex>
#include <regex>
#include <sstream>
#include <string>
#include <vector>

#include <mosquitto.h>

#include <signal.h>
#include <time.h>
#include <sys/types.h>
#include <unistd.h>

#include "influxdb.h"
#include "log.h"
log_t log_level = INFO;

static bool term = false;
static std::string argv0;

// MQTT broker hostname.
static std::string mqtt_host("localhost");
static int mqtt_port = 1883;

// InfluxDB hostname and database name.
static std::string influxdb_host("localhost:8086");
static std::string influxdb_db("mydb");

// This gets populated by optarg or /etc/app-id.
static std::string id("");

// Default field name for measurement values.
static std::string default_field("value");

// Default timestamp precision.
static std::string precision("ms");

// The schema we understand. Also used as an anchor point for our
// subscriptions.
static std::string home("v1/");

// Manages our connection to InfluxDB.
static influxdb db;

// Translates a mosquitto error code into a text string.
static const char* moserrno(int rc) { return mosquitto_strerror(rc); }

// Removes newlines and spaces from @str.
static void cleanup(std::string& str)
{
       std::size_t pos;
       do {
              pos = str.find_first_of(" \r\n");
              if (pos != std::string::npos)
                     str.erase(pos, 1);
       } while (pos != std::string::npos);
}

// signal(1) handler, so that we can exit cleanly.
static volatile int run = 1;
void on_signal(int sig)
{
       switch (sig)
       {
       case SIGTERM:
       case SIGINT:
              run = 0;
              break;
       }

       switch (sig)
       {
       case SIGHUP:
              // TODO: reload config, etc.
              info("SIGHUP received");
              break;
       }
}

// Subscribes to @topic.
static int sub(struct mosquitto* mosq,
               const std::string& topic,
               int qos = 1)
{
       int rc;
       std::string t = topic;

       // Anchor topic under <home> unless otherwise specified.
       if (t.find(home) == std::string::npos)
              t = home + topic;

       rc = mosquitto_subscribe(mosq, NULL, t.c_str(), qos);

       std::stringstream ss;
       ss << "mosquitto_subscribe(\"" << t << "\"): "
          << moserrno(rc);
       trace(ss);

       return rc;
}

static void on_connect(struct mosquitto *mosq, void *obj, int rc)
{
       // ref: http://docs.oasis-open.org/mqtt/mqtt/v3.1.1/mqtt-v3.1.1.html
       trace(std::string("on_connect(): ") + mosquitto_connack_string(rc));
       if (rc)
              exit(1);

       // Subscribe to the "everything" topic.
       rc = sub(mosq, home + "#", 1);
}

static void on_disconnect(struct mosquitto* mosq, void* obj, int rc)
{
       trace(std::string("on_disconnect(): ") + mosquitto_connack_string(rc));
}

static void on_subscribe(struct mosquitto* mosq, void* obj,
                         int mid, int qos_count, const int* granted_qos)
{
       trace(std::string("on_subscribe(): mid=") + std::to_string(mid)
             + std::string(" qos_count=") + std::to_string(qos_count));
}

static void send(std::stringstream& ss)
{
       db.send(ss);
       trace(ss);
}

// Returns the current time in nanoseconds.
static uint_least64_t get_timestamp(void)
{
       struct timespec tnow;
       clock_gettime(CLOCK_REALTIME, &tnow);
       uint_least64_t ret = tnow.tv_sec;
       if (precision == "s")
              return ret;
       if (precision == "ms")
              return ret * 1000UL + (tnow.tv_nsec / 1000000UL);
       if (precision == "us")
              return ret * 1000000UL + (tnow.tv_nsec / 1000UL);
       if (precision == "ns")
              return ret * 1000000000UL + tnow.tv_nsec;
       return 0;
}

// Breaks up @topic into tokens.
static std::vector<std::string> tokenize_topic(const char* topic)
{
       std::vector<std::string> ret;
       char** topics;
       int ntopics;
       if (MOSQ_ERR_SUCCESS == mosquitto_sub_topic_tokenise(topic, &topics, &ntopics))
       {
              for (int n = 0; n < ntopics; n++)
                     ret.push_back(topics[n]);
              mosquitto_sub_topic_tokens_free(&topics, ntopics);
       }
       return ret;
}

// Breaks up @msg into tokens.
static std::vector<std::string> tokenize_payload(const void* msg, const size_t len)
{
       // Throw out overly-long messages.
       if (len >= 1024)
              return std::vector<std::string>{};

       const std::regex re(R"([\s]+)");
       std::string s((const char*)msg, len);
       std::sregex_token_iterator it{s.begin(), s.end(), re, -1};
       std::vector<std::string> tok{it, {}};

       // Reject the message if it contains unprintable characters.
       if (std::all_of(s.begin(), s.end(), []( char c ) { return !std::isprint(c); }))
              return std::vector<std::string>{};

       // Split the message up into tokens.
       tok.erase(std::remove_if(tok.begin(),
                                tok.end(),
                                [](std::string const& s) {return s.size() == 0;}),
                 tok.end());
       return tok;
}

// Drops @tok if it isn't already labeled, as is required of tag fields.
static std::string to_tag(const std::string tok)
{
       std::string tag;
       if (tok.find_first_of("=") != std::string::npos)
              tag = tok;
       return tag;
}

// Applies @label to @tok, if it isn't labeled already.
static std::string to_field(const std::string tok, const std::string& label = default_field)
{
       std::string field;
       if (tok.find_first_of("=") != std::string::npos)
              field = tok;
       else
              field = label + "=" + tok;
       return field;
}

// Drops @tok if it isn't a timestamp, i.e. it contains an '='.
static std::string to_timestamp(const std::string tok)
{
       std::string ts;
       if (tok.find_first_of("=") == std::string::npos)
              ts = tok;
       return ts;
}

// Measurement names must differ from field labels.
static std::string to_measurement(const std::string tok, const std::string& label = default_field)
{
       std::string measurement;
       if (tok != label)
              measurement = tok;
       return measurement;
}


// Called by mosquitto when we receive a message on a subscribed topic. We
// reformat and send the payload to InfluxDB.
//
// Remember, MQTT topics are NOT MESSAGE QUEUES! Each publication is a
// standalone entity: the complete topic name is just a fancy-looking label,
// and the message is just the value to associate with that label.
//
static void on_message(struct mosquitto* mosq, void* obj,
                       const struct mosquitto_message* msg)
{
       uint_least64_t tnow = get_timestamp();
       std::vector<std::string> topics, tok;
       std::stringstream ss;
       std::string measurement;
       std::string field;
       std::string timestamp;

       // Always tag measurements with our unique host id, to prevent polluting
       // the database with ambiguous data.
       std::string tag("host=" + id);

       // If there's no message payload, give up. We don't care about expiring
       // of retained topics.
       if (msg->payloadlen <= 0)
              return;

       // Tokenize each topic by level.
       topics = tokenize_topic(msg->topic);
       if (topics.size() <= 0)
              return;

       // Make sure the schema is a format we understand.
       if (topics.size() < 2 || topics[0] != "v1")
              return;

       // Tokenize the payload.
       tok = tokenize_payload(msg->payload, msg->payloadlen);
       if (tok.size() < 1)
              return;

       // The last field of the topic name is the measurement name, regardless
       // of the number of tokens found.
       //
       // "v1/<measurement>", "v1/<foo>/<measurement>", etc.
       measurement = to_measurement(topics[topics.size() - 1]);
       if (measurement.length() <= 0)
              return; // Measurements must have names.

       // Tag the measurement with the whole MQTT topic name.
       tag += ",mqtt=\"";
       tag += (const char*)msg->topic;
       tag += "\"";

       // If the payload has only one token, it's a simple value.
       if (tok.size() == 1)
              field = to_field(tok[0]);

       // If the payload has two tokens, then it's either a key-value with
       // timestamp, or labeled tags and fields with no timestamp.
       else if (tok.size() == 2)
       {
              timestamp = to_timestamp(tok[1]);
              if (timestamp.length() <= 0)
              {
                     // The second token wasn't a timestamp, it must be a field
                     // set. That makes the first token a tag set.
                     std::string tmp = to_tag(tok[0]);
                     if (tmp.length() <= 0)
                            return; // Tags must have labels.
                     tag += "," + tmp;

                     field = to_field(tok[1]);
              }
              else
              {
                     // The second token was a timestamp, so the first token
                     // must be a field set.
                     field = to_field(tok[0]);
              }
       }

       else if (tok.size() >= 3) // (We silently ignore fields > 3, tho.)
       {
              // The payload contains a tag set, a field set, and a timestamp.

              // If we can't parse the timestamp, drop the message.
              timestamp = to_timestamp(tok[2]);
              if (timestamp.length() <= 0)
                     return;

              std::string tmp = to_tag(tok[0]);
              if (tmp.length() <= 0)
                     return; // tags must have labels
              tag += "," + tmp;

              field = to_field(tok[1]);
       }

       // If no timestamp was provided, use ours.
       if (timestamp.length() <= 0)
              timestamp = std::to_string(tnow);

       // Assemble and send the message.
       ss << measurement << "," << tag << " " << field << " " << timestamp;
       send(ss);
       return;
}

int main(int argc, char *argv[])
{
       argv0 = argv[0];

       int opt;
       while ((opt = getopt(argc, argv, "c:C:H:h:I:K:m:p:q:ridtv")) != -1)
       {
              switch (opt)
              {
              case 'c': // client ID
                     id = optarg;
                     break;
              case 'H': // home override (debug only)
                     home = optarg;
                     break;
              case 'h': // broker hostname/IP
                     mqtt_host = optarg;
                     break;
              case 'p': // timestamp precision
                     precision = optarg;
                     break;
              case 't': // SIGTERM override (usually a bad idea)
                     term = true;
                     break;
              case 'v': // verbose output
                     log_level = TRACE;
                     break;
              case 'd': // debugging output
                     log_level = DEBUG;
                     break;
              default: /* '?' */
                     std::cout << "TODO: Usage: " << (const char*)argv[0] << std::endl;
                     exit(EXIT_FAILURE);
              }
       }

       // Read the app-id, if we don't have an identifier already.
       if (id.length() == 0)
       {
              std::ifstream in("/etc/app-id");
              std::stringstream sstr;
              in >> sstr.rdbuf();
              if (sstr.str().length() != 0)
                     id = sstr.str();
              else
                     id = "<undefined>";
       }

       // Make sure our id is clean.
       cleanup(id);

       // Display our timestamp precision, in case somebody cares.
       trace(std::string("timestamp precision: ") + precision);

       struct mosquitto *mosq;
       int rc = 0;

       // Specify our own signal handlers.
       signal(SIGINT, on_signal);
       signal(SIGHUP, on_signal);
       if (term)
              signal(SIGTERM, on_signal);

       mosquitto_lib_init();
       {
              int mmaj, mmin, mrev;
              mosquitto_lib_version(&mmaj, &mmin, &mrev);
              std::stringstream ss;
              ss << "libmosquitto: " << mmaj << "." << mmin << "." << mrev;
              debug(ss);
       }

       mosq = mosquitto_new(id.c_str(), true, 0);
       if (!mosq)
       {
              fatal("mosquitto_new(): NULL");
              mosquitto_lib_cleanup();
              exit(1);
       }
       
       // Set up callbacks.
       mosquitto_connect_callback_set(mosq, on_connect);
       mosquitto_disconnect_callback_set(mosq, on_disconnect);
       mosquitto_subscribe_callback_set(mosq, on_subscribe);
       mosquitto_message_callback_set(mosq, on_message);

       // Connect to broker.
       rc = mosquitto_connect(mosq, mqtt_host.c_str(), mqtt_port, 10);
       if (rc)
       {
              fatal(std::string("mosquitto_connect(): ") + moserrno(rc));
              exit(1);
       }

       // Setup the InfluxDB connection.
       db.set_host(influxdb_host);
       db.set_database(influxdb_db);
       db.set_precision(precision);

       while (run)
       {
              // Main loop.
              rc = mosquitto_loop(mosq, -1, 1);
              if (run && rc)
              {
                     info(std::string("connection error: ") + moserrno(rc));
                     sleep(1);
                     mosquitto_reconnect(mosq);
              }
       }

       mosquitto_destroy(mosq);
       mosquitto_lib_cleanup();

       return rc;
}


